/* 
 * 2011 at the University of applied sciences Augsburg
 * as written by Claus Hoffmann for his bachelorthesis "the singing CANVAS"
 * this code is published under creative commons attribution non-commercial share-alike
 * You may rewrite and use this code as much as you like, as long as it is not for
 * commercial purposes.
 */

package main;


import fullscreen.FullScreen;
import graphic.Brush;
import graphic.CanvasAnalysis;
import graphic.ColorManagement;
import graphic.DirectionManagement;
import graphic.HitBorderEvent;
import graphic.HitBorderEventListener;
import gui.Setup;
import gui.StartAnimation;
import javax.swing.JFileChooser;

import audio.MIDI_Input;
import java.awt.Container;
import java.awt.Frame;
import java.awt.Menu;
import java.awt.MenuBar;
import java.awt.MenuItem;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.io.File;
import processing.core.*;
import promidi.*;

/*
 * the singing CANVAS
 * The main class. Here everything gets controlled.
 */


public class Main extends PApplet implements HitBorderEventListener{
	
	private static final long serialVersionUID = 5853628603517515934L;
	
	// STATICS THAT ALLOW FINETUNING
	private int STAKKATO_TIME = 13;
	private int DURATION_TRESHOLD_SMALL = 150;
	private int HPS_TRESHOLD_SMALL = 19;
	private int DURATION_TRESHOLD_LARGE = 250;
	private int HPS_TRESHOLD_LARGE = 29;
	// -----------------------------
	
	//PDF Writer
	private boolean pdfOut = false;
	final JFileChooser fc = new JFileChooser();
	private String filename = "beginRecordFile";
	
	//Fullscreensupport
	private Frame thisFrame;
	
	private FullScreen fullscreen;
	
	//audio analysis
	private MIDI_Input midiInput; 
	
	//graphic analysis
	private CanvasAnalysis canvasAnalysis;
	private ColorManagement colorManagement;
	private DirectionManagement directionManagement;
	
	//GUI-Setup
	private Setup primarySetup;
	private StartAnimation startAnimation;
	
	//flags
	private boolean upAndRunning = true;
	private boolean notesArePlayed = false;
	private boolean paintingInProgress = false;
	
	//Is there still Input?
	private long startingPointForPlayedNotes;
	private long TIME_SPAN_FOR_PLAYED_NOTES = 700; //how long do we wait for further input?
	
	private boolean firstNote = true;
	
	
	private int stakkatoCounter1 = 0;
	private int stakkatoCounter2 = 0;
	
	//Brush and Brushessetup
	private int[] initialColor = {255,255,255,255}; //we start our brush with a white color
	private Brush b1;
	private Brush b2;

	
	public Main(){
		
	}
	
	//main method for application export
	public static void main(String args[])
    {
      PApplet.main(new String[] { main.Main.class.getName() });
    }
	
	public void setup(){
		
		
		//setup Canvas
		thisFrame = findFrame();
		thisFrame.setTitle("the singing Canvas");
		size(440, 700);
		frameRate(30);
		
		//setup fullscreen mode
		fullscreen = new FullScreen(this);
		fullscreen.setShortcutsEnabled(true);
		thisFrame.setIconImage( getToolkit().getImage("sketch.ico") );
		
		
		//setup Menubar
		System.setProperty("apple.laf.useScreenMenuBar", "true");
		MenuBar menuBar = new MenuBar();
		Menu viewMenu = new Menu("View");
		menuBar.add(viewMenu);
		MenuItem fullScreenAction = new MenuItem("Fullscreen   cmd+f");
		viewMenu.add(fullScreenAction);
		fullScreenAction.addActionListener(new ActionListener(){
			public void actionPerformed(ActionEvent arg0){
					fullscreen.setFullScreen(true);
			}
		 });
		frame.setMenuBar(menuBar);
		
		
		//setup audio analysis
		midiInput = new MIDI_Input(this);
		primarySetup = new Setup(this);
		primarySetup.setMidiInputs(midiInput.getDeviceNames());
		primarySetup.hideGUI();
		
		initialize();
	}
	
	private void initialize(){
		//set everything to default
		background(255);
		beginRecord(PDF, filename+".pdf");
		primarySetup.showGUI();

		colorManagement = new ColorManagement();	
		canvasAnalysis = new CanvasAnalysis(this, 20, 20);
		directionManagement = new DirectionManagement();
		startAnimation = new StartAnimation(this);
		midiInput.initialize();
		firstNote = true;
		
		
		smooth();
		b1 = new Brush(this, initialColor, this.width/2, this.height/2, (float)0, (float)0, 10, "BRUSH");
		b2 = new Brush(this, initialColor, this.width/2, this.height/2, (float)0, (float)0, 10, "FEATHER");
		
		b1.addHitBorderEventListener(this);
		b2.addHitBorderEventListener(this);
		
	}
	
	public void draw() {
		if(!primarySetup.isVisible()){ //check if we are in painting mode already
			paintingInProgress = startAnimation.getReadyState(); //check if we have enough data to start
			checkIfNotesArePlayed(); //check if we are still playing
			
			// everything fine? map our data onto our brushes!
			if(paintingInProgress && notesArePlayed){ 
				b1.setSpeed(midiInput.getHpsCounterData());
				b1.setAlpha(midiInput.getVolumeAnalyserData());
				b1.setColor(colorManagement.getColor(midiInput.getNoteDetectionData(),midiInput.getPitch(),midiInput.getOverallTune()));
				
				b1.setThickness(midiInput.getNoteDurationsData(),midiInput.getPitch() );
				directionManagement.adjustSteps(midiInput.getHpsCounterData());
				
				float dirX = directionManagement.analyseDirectionX(midiInput.getNoteTurningPointsTurns(),midiInput.getNoteTurningPointsMainDirection());
				float dirY = directionManagement.analyseDirectionY(midiInput.getNoteTurningPointsTurns(),midiInput.getNoteTurningPointsMainDirection());
				b1.setCurveDirection(dirX, dirY);
				//... and move the brush
				b1.moveBrush();
				//-----------------------------------------
				b2.setSpeed(midiInput.getHpsCounterData());
				b2.setAlpha(midiInput.getVolumeAnalyserData());
				b2.setColor(colorManagement.getColor(midiInput.getNoteDetectionData(),midiInput.getPitch(),midiInput.getOverallTune()));
				b2.setCurveDirection(dirY, dirX);
				b2.setThickness(0,0);
				//... and move the brush
				b2.moveBrush();
				
				
				
				
				if(midiInput.getNoteDurationsData() < DURATION_TRESHOLD_SMALL && midiInput.getHpsCounterData() > HPS_TRESHOLD_SMALL){	
					if(stakkatoCounter1 == 0){
						b1.setNewPosition(canvasAnalysis.getNewPosition(false));
					}
					stakkatoCounter1++;
					if(stakkatoCounter1 == STAKKATO_TIME){
						stakkatoCounter1 = 0;
					}
					
				}
				if(midiInput.getNoteDurationsData() < DURATION_TRESHOLD_LARGE && midiInput.getHpsCounterData() > HPS_TRESHOLD_LARGE){	
					if(stakkatoCounter2 == 0){
						//b2.setNewPosition(canvasAnalysis.getNewPosition(true));
						b2.setNewPosition(b1.getCurrentPosition());
					}
					stakkatoCounter2++;
					if(stakkatoCounter2 == STAKKATO_TIME){
						stakkatoCounter2 = 0;
					}	
				}
			
			}else{
				startAnimation.setAnimStage(midiInput.getReadyState());
			}
		}else{
			startAnimation.setAnimStage(0);
		}
	}
	
	public void noteOn(Note note, int device, int channel){
		if(!primarySetup.isVisible()){
			if(upAndRunning){ 	//Programm started successfully?
				if(!pdfOut){	//No printing in Progress?
					midiInput.noteOn(note, device, channel);	//pass note to the audioanalysis
					notesArePlayed = true;						//tell the System we are still playing
					startingPointForPlayedNotes = System.currentTimeMillis();
					if(firstNote){								//if this is our firstNote, reset everything
						firstNote = false;
						startAnimation.startTheAnimation();
					}
				}
			}
		}
	}
	public void noteOff(Note note, int device, int channel){
		if(upAndRunning){
			if(!pdfOut){
				midiInput.noteOff(note);
			}
		}
	}
	
	
	//catch all keyboard input
	public void keyPressed(){
		if(key == ESC){
			key = 0;
			if(fullscreen.isFullScreen()){
				fullscreen.setFullScreen(false);
			}
		}
		if(keyCode == ENTER){
			startButton(0);
		}
		if(keyCode == TAB){
			primarySetup.changeInputFocus();
		}
		if(key == 'p'){
			if(!primarySetup.isVisible()){	
				printButton(0);
			}
		}
	}
	
	private void checkIfNotesArePlayed(){
		if(System.currentTimeMillis() - startingPointForPlayedNotes > TIME_SPAN_FOR_PLAYED_NOTES){
			notesArePlayed = false;
		}
	}
	
	public Frame findFrame() {		
	      Container f = this.getParent();
	      while (!(f instanceof Frame) && f!=null)
	            f = f.getParent();
	      return (Frame) f;
	}

	public void hitBorderEventOccurred(HitBorderEvent evt) {
		//our brush hit a border, start at one of the lightest spots
		if(evt.getMode()=="BRUSH"){
			((Brush)evt.getSource()).setNewPosition(canvasAnalysis.getNewPosition(false));
		}else{
			((Brush)evt.getSource()).setNewPosition(canvasAnalysis.getNewPosition(true));
		}
		
	}
	
	public void startButton(int value){
		if(primarySetup.isVisible()){
			if(primarySetup.getRadioState() == -1){
				primarySetup.setMidiInputs(midiInput.getDeviceNames());
			}else{
				midiInput.openMidiInput(primarySetup.getRadioState());
				if(primarySetup.checkData()){
					endRecord();
					//Hopefully this is platformindipendent!
					String fileToDeletePath = dataPath(this.sketchPath+System.getProperty("file.separator") +filename+".pdf");
					File f = new File(fileToDeletePath);
					if (f.exists()) {
					    f.delete();
					}
					filename = primarySetup.getFilename();
					beginRecord(PDF, filename+".pdf");
					primarySetup.hideGUI();
				}
			}
		}
	}
	
	public void printButton(int value){
		strokeWeight(1);
		endRecord();
		filename="beginRecord";
		initialize();
	}

}
