/* 
 * 2011 at the University of applied sciences Augsburg
 * as written by Claus Hoffmann for his bachelorthesis "the singing CANVAS"
 * this code is published under creative commons attribution non-commercial share-alike
 * You may rewrite and use this code as much as you like, as long as it is not for
 * commercial purposes.
 */

package gui;

import java.text.SimpleDateFormat;
import java.util.Calendar;

import controlP5.*;
import processing.core.PApplet;
import processing.pdf.PGraphicsPDF;

/*
 * the Setup is the Interface that is shown right at the start.
 * Here we get all the important information:
 * - what is going to be played?
 * - who is going to play it?
 * - from which midiInput is the data going to be coming from?
 * 
 * The Interface was done with a library called controlP5.
 * further information at http://www.sojamo.de/libraries/controlP5
 */


public class Setup {
	private ControlP5 controlP5;
	private PApplet parent;
	
	//all the Interface elements
	private Textlabel theSingingCanvas;
	private Textlabel titleLabel;
	private Textfield titleTextfield;
	private Textlabel artistLabel;
	private Textfield artistTextfield;
	private Textlabel chooseMidiLabel;
	private Textlabel printTextfield;
	private Textlabel warningText;
	private RadioButton radioButtons;
	private Button startButton;
	//-------------------
	
	//the colours used and the positioning
	private int TEXTFIELD_WIDTH = 150;
	private int TEXTFIELD_HEIGHT = 14;
	
	
	private int DARK= 20;
	private int MEDIUM = 100;
	private int LIGHT = 200;
	
	private int PADDING_TOP = 400;
	private int PADDING_LEFT = (440-TEXTFIELD_WIDTH)/2;
	
	private String FONT = "Times New Roman";
	
	private int numberOfMidiDevices;
	
	
	
	//The title that is initially written in the title field
	@SuppressWarnings("unused")
	private String START_TITLE = "Testmalen_Theme";
	
	public Setup(PApplet parent){
		this.parent = parent;
		controlP5 = new ControlP5(parent);
		
		

		
		
		// SETTING UP ALL OUR ELEMENTS START
		theSingingCanvas = controlP5.addTextlabel("theSingingCanvas","the singing CANVAS",PADDING_LEFT,PADDING_TOP);
		theSingingCanvas.valueLabel().setColor(parent.color(DARK));
		
		theSingingCanvas.draw(parent);
		

		
		titleLabel = controlP5.addTextlabel("titleLabel","this canvas will be",PADDING_LEFT,PADDING_TOP+25);
		titleLabel.valueLabel().setColor(parent.color(MEDIUM));
		titleLabel.draw(parent);
		
		titleTextfield = controlP5.addTextfield("title",PADDING_LEFT,PADDING_TOP+40,TEXTFIELD_WIDTH,TEXTFIELD_HEIGHT);
		
		titleTextfield.valueLabel().style().movePadding(-4, 0, 0, 0);
		titleTextfield.setColorBackground(255);
		titleTextfield.setColorForeground(LIGHT);
		titleTextfield.valueLabel().setColor(DARK);
		titleTextfield.setColorActive(DARK);
		titleTextfield.setFocus(true);
		titleTextfield.setValue("");
		titleTextfield.setAutoClear(false);
		
		artistLabel = controlP5.addTextlabel("artistLabel","as played by",PADDING_LEFT,PADDING_TOP+65);
		artistLabel.valueLabel().setColor(parent.color(MEDIUM));
		artistLabel.draw(parent);
		
		artistTextfield = controlP5.addTextfield("artist",PADDING_LEFT,PADDING_TOP+80,TEXTFIELD_WIDTH,TEXTFIELD_HEIGHT);
		
		artistTextfield.valueLabel().style().movePadding(-4, 0, 0, 0);
		artistTextfield.setColorBackground(255);
		artistTextfield.setColorForeground(LIGHT);
		artistTextfield.valueLabel().setColor(DARK);
		artistTextfield.setColorActive(DARK);
		artistTextfield.setFocus(false);
		artistTextfield.setValue("");
		artistTextfield.setAutoClear(false);
		
		
		chooseMidiLabel = controlP5.addTextlabel("chooseMidiLabel","choose your input:",PADDING_LEFT,PADDING_TOP+105);
		chooseMidiLabel.valueLabel().setColor(parent.color(MEDIUM));
		chooseMidiLabel.draw(parent);
		
		warningText = controlP5.addTextlabel("warningText","no Input available",PADDING_LEFT,PADDING_TOP+120);
		warningText.valueLabel().setColor(parent.color(200,0,0));
		warningText.draw(parent);
		warningText.hide();

		radioButtons = controlP5.addRadioButton("radioButton",PADDING_LEFT,PADDING_TOP+120);
		radioButtons.setNoneSelectedAllowed(false);
		radioButtons.setColorForeground(parent.color(100));
		radioButtons.setColorActive(parent.color(MEDIUM));//activeColor des RadioButtons
		radioButtons.setColorBackground(parent.color(255));//inactiveColor des RadioButtons
		radioButtons.setColorLabel(parent.color(DARK)); // Schriftfarbe
		radioButtons.setItemHeight(TEXTFIELD_HEIGHT);    //22
		radioButtons.setItemWidth(TEXTFIELD_HEIGHT);
		radioButtons.setItemsPerRow(1);
		radioButtons.setSpacingRow(4);
		
		// SETTING UP ALL OUR ELEMENTS END
		//our fonts used
		
		String[] fonts = PGraphicsPDF.listFonts();
		for(int font = 0; font<fonts.length; font++){
			if(fonts[font].equals(FONT)){
				ControlFont h1 = new ControlFont(parent.createFont(FONT,815), 17);
				ControlFont p = new ControlFont(parent.createFont(FONT,815), 11);
				controlP5.setControlFont(p);
				titleTextfield.valueLabel().setControlFont(p);
				theSingingCanvas.valueLabel().setControlFont(h1);
				artistTextfield.valueLabel().setControlFont(p);
				break;
			}
		}
		
		controlP5.setAutoDraw(true);
		controlP5.show();
		
		
	}
	private void addToRadioButton(RadioButton theRadioButton, String theName, int theValue ) {
		  //adding a radio button and styling it
		  Toggle t = theRadioButton.addItem(theName,theValue);
		  t.captionLabel().setColorBackground(parent.color(230));
		  t.captionLabel().style().movePadding(3,0,0,2);
		  t.captionLabel().style().moveMargin(-2,100,0,0);
		  t.valueLabel().style().marginTop=15;
		  t.captionLabel().style().backgroundHeight = TEXTFIELD_HEIGHT-3;
		  t.captionLabel().style().backgroundWidth = TEXTFIELD_WIDTH - TEXTFIELD_HEIGHT -6;
		  CVector3f pos = t.position();
		  CVector3f absoloutePos = t.absolutePosition();
		  //creating a border box
		  RadioRect radioBox = new RadioRect(absoloutePos.x + pos.x,absoloutePos.y+pos.y, TEXTFIELD_HEIGHT-1, TEXTFIELD_HEIGHT-1);
		  radioBox.post();
		  controlP5.controlWindow.addCanvas(radioBox);
	}

	public void setMidiInputs(String[] inputDevices){
		
		numberOfMidiDevices = inputDevices.length;
		for(int x = 0; x<inputDevices.length; x++){
			addToRadioButton(radioButtons,inputDevices[x],x);
		}
		printTextfield = controlP5.addTextlabel("printTextfield","press <p> to print when finished",PADDING_LEFT,(PADDING_TOP+120+(numberOfMidiDevices+1)*(TEXTFIELD_HEIGHT+2)));
		printTextfield.valueLabel().setColor(parent.color(MEDIUM));
		printTextfield.draw(parent);
		startButton = controlP5.addButton("startButton",0,PADDING_LEFT+27,(PADDING_TOP+120+(numberOfMidiDevices+2)*(TEXTFIELD_HEIGHT+2)),TEXTFIELD_WIDTH -60,TEXTFIELD_HEIGHT);
		startButton.captionLabel().style().marginLeft = 17;
		startButton.captionLabel().style().movePadding(-1,0,0,7);
		startButton.setColorBackground(parent.color(LIGHT));
		startButton.setColorForeground(MEDIUM);
		startButton.setCaptionLabel("start");
		startButton.setColorActive(parent.color(DARK));
		
		
		if(numberOfMidiDevices == 0){
			//if we don't have any MIDI Inputs we post a warning and hide the startbutton
			startButton.hide();
			printTextfield.valueLabel().set("please make your Midi-Input available and restart");
			warningText.show();
		}else{
			radioButtons.activate(numberOfMidiDevices -1); // The initially selected radiobutton
		}
		
		
		
	}
	
	public int getRadioState(){
		int inputDevice = -1;
		for(int x = 0; x < numberOfMidiDevices; x++){
			if(radioButtons.getState(x)){
				inputDevice = x;
				break;
			}
		}
		return inputDevice;
	}

	
	public void changeInputFocus(){
		//on pressing tab, we change the focus of the textfields
		if(titleTextfield.isFocus()){
			titleTextfield.setFocus(false);
			artistTextfield.setFocus(true);
		}else if(artistTextfield.isFocus()){
			artistTextfield.setFocus(false);
			titleTextfield.setFocus(true);
		}
		if(!titleTextfield.isFocus() && !artistTextfield.isFocus()){
			titleTextfield.setFocus(true);
		}
		
	}
	
	public boolean checkData(){
		//only if we have information in the textfields we start
		boolean ret = true;
		if(titleTextfield.getText() == ""){
			ret = false;
		}
		if(artistTextfield.getText() == ""){
			ret = false;
		}
		return ret;
	}
	
	public String getFilename(){
		Calendar cal = Calendar.getInstance();
		SimpleDateFormat sdf = new SimpleDateFormat("yyyy_MM_dd");
		sdf.format(cal.getTime());

		return sdf.format(cal.getTime())+"_"+titleTextfield.getText()+"_by_"+artistTextfield.getText();
	}
	
	public void hideGUI(){
		controlP5.setAutoDraw(false);
		controlP5.hide();
	}
	public void showGUI(){
		controlP5.setAutoDraw(true);
		controlP5.show();
	}
	
	public boolean isVisible(){
		return controlP5.isVisible();
	}
	
	
	class RadioRect extends ControlWindowCanvas {
		// this class enables a border around the radioButtons
		private float posX;
		private float posY;
		private int width;
		private int height;
		
		public RadioRect(float x, float y, int width, int height){
			this.posX = x;
			this.posY = y;
			this.width = width;
			this.height = height;
		}

		public void draw(PApplet parent) {
			  parent.stroke(parent.color(LIGHT));
			  parent.noFill();
			  parent.rect(posX,posY,width,height);
		}
	}

}
