/* 
 * 2011 at the University of applied sciences Augsburg
 * as written by Claus Hoffmann for his bachelorthesis "the singing CANVAS"
 * this code is published under creative commons attribution non-commercial share-alike
 * You may rewrite and use this code as much as you like, as long as it is not for
 * commercial purposes.
 */

package graphic;

import javax.swing.event.EventListenerList;

import processing.core.*;

/*
 * The Brush class. Basically it just draws a line from the old position to the new position
 * after every property has been updated. 
 */
public class Brush{
	
	// STATICS THAT ALLOW FINETUNING
	private int SPEED_DECREASE_BRUSH = 45; //slow down the drawing // more is slower
	private int SPEED_DECREASE_FEATHER = 30; //slow down the drawing // more is slower
	private int BASIC_ALPHA = 160; // make it all a little bit more transparent, the higher the number, the less transparent
	private float THICKNESS_DECREASE = 28; //the more, the less thick
	private float THICKNESS_TRANSLATION_STEPS = 40;
	private float PITCH_START = 80; // PITCH_START - pitch gets added to thickness; 
	
	private float FEATHER = 2;
	private float BASIC_FEATHER_WIDTH = 1;
	private float MAX_FEATHER_THICKNESS = 10;
	private int FEATHER_BLACKENER = 50;
	private float MIN_SPEED = (float)0.4;
	// -----------------------------
	
	private String mode;
	
	
	protected EventListenerList listenerList = new EventListenerList();
	private HitBorderEvent hitBorderEvent;
	
	private PApplet parent;
	private float positionX;
	private float positionY;
	
	private float oldPositionX;
	private float oldPositionY;
	
	private float newThickness;
	private float thickness;
	private float thicknessSteps = 0;

	private float dirX;
	private float dirY;
	
	private float directionX;
	private float directionY;
	
	private float curveDirX = 0;
	private float curveDirY = 0;
	
	private int[] color = new int[4];
	
	private float speed;
	
	public Brush(PApplet parent, int[] brushColor, int positionX, int positionY, float dirX, float dirY, int thickness, String mode){
		this.parent = parent;
		this.positionX = positionX;
		this.positionY = positionY;
		this.oldPositionX = positionX;
		this.oldPositionY = positionY;
		this.thickness = thickness;
		this.color = brushColor;
		this.dirX = dirX;
		this.dirY = dirY;
		this.mode = mode;
		hitBorderEvent = new HitBorderEvent(this, mode);
	}
	public void moveBrush(){
		oldPositionX = positionX;
		oldPositionY = positionY;
		
		directionX = dirX + curveDirX;
		directionY = dirY + curveDirY;
		
		positionX += directionX*speed;
		positionY += directionY*speed;
		
		if(mode =="BRUSH"){
			translateThickness();
		}
		if(mode == "FEATHER"){
			thicknessSteps = THICKNESS_TRANSLATION_STEPS;
			thickness = newThickness;
		}
		parent.strokeWeight(thickness);
		

		parent.smooth();
		parent.stroke(color[0],color[1],color[2],color[3]);
		parent.line(oldPositionX, oldPositionY, positionX, positionY);
		if(positionX < 0 || positionX >parent.width){
			dirX = -dirX;
			this.fireHitBorderEvent(hitBorderEvent);
		}
		if(positionY < 0 || positionY > parent.height){
			this.fireHitBorderEvent(hitBorderEvent);
			dirY = -dirY;
		}
	}
	
	public void setSpeed(float newSpeed){
		
		if(mode == "BRUSH"){
			this.speed = newSpeed/SPEED_DECREASE_BRUSH;
			if(speed<=0){
				speed = MIN_SPEED;
			}
		}if(mode =="FEATHER"){
			this.speed = newSpeed/SPEED_DECREASE_FEATHER;
			if(speed<=0){
				speed = MIN_SPEED;
			}
		}
	}
	public void setAlpha(int alpha){
		if(mode=="BRUSH"){
			this.color[3] = (int) ((alpha*BASIC_ALPHA/127) + speed*7);
		}else if(mode=="FEATHER"){
			this.color[3] = (int) ((alpha*BASIC_ALPHA/30) + speed*7);
		}
	}
	public void setThickness(float thickness, float pitch){
		
		if(mode =="BRUSH"){
			this.newThickness = (thickness/THICKNESS_DECREASE)+(PITCH_START-pitch);	//brush mode
		}else if(mode== "FEATHER"){// feather mode
			this.newThickness = (float) ((FEATHER+BASIC_FEATHER_WIDTH)+ (Math.abs(directionX)-Math.abs(directionY))*FEATHER);
			if(thickness > MAX_FEATHER_THICKNESS){
				thickness = MAX_FEATHER_THICKNESS;
			
			}
		}else{
			this.newThickness = thickness;
		}
		if(newThickness<= 0){
			newThickness = (float) 0.2;
		}
		thicknessSteps = 0;
	}
	
	public void setColor(int[] newColor){
		if(mode=="BRUSH"){
			this.color[0] = newColor[0];
			this.color[1] = newColor[1];
			this.color[2] = newColor[2];
		}else if(mode =="FEATHER"){
			int col = (newColor[0]+newColor[1]+newColor[2])/3;
			col -= FEATHER_BLACKENER;
			if(col < 0){
				col = 0;
			}
			if(col>150){
				col=150;
			}
			this.color[0] = col;
			this.color[1] = col;
			this.color[2] = col;
		}else{
			this.color[0] = newColor[0];
			this.color[1] = newColor[1];
			this.color[2] = newColor[2];
		}
	}
	
	public void setCurveDirection(float curveDirectionX, float curveDirectionY){
		this.curveDirX = curveDirectionX;
		this.curveDirY = curveDirectionY;
	}
	
	public void addHitBorderEventListener(HitBorderEventListener listener) {
	    listenerList.add(HitBorderEventListener.class, listener);
	  }
	public void removeHitBorderEventListener(HitBorderEventListener listener) {
	    listenerList.remove(HitBorderEventListener.class, listener);
	  }
	public void fireHitBorderEvent(HitBorderEvent evt) {
	    Object[] listeners = listenerList.getListenerList();
	    for (int i = 0; i < listeners.length; i = i+2) {
	      if (listeners[i] == HitBorderEventListener.class) {
	        ((HitBorderEventListener) listeners[i+1]).hitBorderEventOccurred(evt);
	      }
	    }
	  }
	public void setNewPosition(int[] newPosition) {
		this.positionX = newPosition[0];
		this.positionY = newPosition[1];
	}
	
	private void translateThickness(){
		if(thickness != newThickness){
			thicknessSteps++;
			if(thicknessSteps != THICKNESS_TRANSLATION_STEPS){
				thickness += (newThickness - thickness)/(THICKNESS_TRANSLATION_STEPS - thicknessSteps);
			}else{
				thickness = newThickness;
			}
		
		}
	}
	
	public int[] getCurrentPosition(){
		int[] ret = {(int)oldPositionX,(int) oldPositionY};
		return ret;
	}

}
