/* 
 * 2011 at the University of applied sciences Augsburg
 * as written by Claus Hoffmann for his bachelorthesis "the singing CANVAS"
 * this code is published under creative commons attribution non-commercial share-alike
 * You may rewrite and use this code as much as you like, as long as it is not for
 * commercial purposes.
 */
package audio;

import processing.core.PApplet;
import promidi.*;
/*
 * The class MIDI_Input opens the MIDIchannel and instantiates all of the
 * audio-analsing tools. Besides that it's a lot of getters and setters. 
 */
public class MIDI_Input{
	
	private PApplet parent;
	private MidiIO midiIO;

	private NoteDetection noteDetection;
	private VolumeAnalyser volumeAnalyser;
	private HpS_Counter hpsCounter;
	private NoteDurations noteDurations;
	private NoteTurningPoints noteTurningPoints;
	private PitchDetection pitchDetection;

	
	public MIDI_Input(PApplet p){
		parent = p;
		midiIO = MidiIO.getInstance(parent);
		midiIO.printDevices();
	}
	
	public void initialize(){
		noteDetection = new NoteDetection();
		volumeAnalyser = new VolumeAnalyser();
		hpsCounter = new HpS_Counter();
		noteDurations = new NoteDurations();
		noteTurningPoints = new NoteTurningPoints();
		pitchDetection = new PitchDetection();
	}
	
	public void openMidiInput(int device){
		midiIO.openInput(device, 0);
	}
	
	public void noteOn(Note note, int device, int channel){
		
		int vol = note.getVelocity(); // Lautstrke
		if(vol != 0){
			noteDetection.analyseNotes(note.getPitch());
			volumeAnalyser.beatGotPlayed(vol);
			hpsCounter.beatGotPlayed();
			noteDurations.noteOnEvent(note.getPitch(), note.getVelocity());
			noteTurningPoints.noteOnEvent(note.getPitch());
			pitchDetection.noteOnEvent(note.getPitch());
			//System.out.println(note.getPitch()+"  "+ note.getVelocity());
		}
	}
	
	public void noteOff(Note note){
		noteDurations.noteOffEvent(note.getPitch());
	}
	
	public void reset(){
		noteDetection.resetAll();
	}
	
	public String getNoteDetectionData(){
		return noteDetection.getCurrentTune();
	}
	public String getOverallTune(){
		return noteDetection.getOverallTune();
	}
	
	public int getVolumeAnalyserData(){
		return volumeAnalyser.getcurrentVolume();
	}
	
	public int getHpsCounterData(){
		return hpsCounter.getCurrentHPS();
	}
	
	public float getNoteDurationsData(){
		return noteDurations.getCurrentAvgLength();
	}
	
	public int getNoteTurningPointsTurns(){
		return noteTurningPoints.getAverageTurns();
	}
	public int getNoteTurningPointsMainDirection(){
		return noteTurningPoints.getMainDirection();
	}
	public int getPitch(){
		return pitchDetection.getCurrentPitch();
	}
	public int getNumberOfInputs(){
		return midiIO.numberOfInputDevices();
	}
	public String[] getDeviceNames(){
		String[] devices = new String[midiIO.numberOfInputDevices()];
		for(int x = 0; x < midiIO.numberOfInputDevices(); x++){
			devices[x] = midiIO.getInputDeviceName(x);
		}
		return devices;
	}

	public int getReadyState(){
		int ret = 0;
		if(noteDetection.getReadyState()){
			ret++;
		}
		if(volumeAnalyser.getReadyState()){
			ret++;
		}
		if(hpsCounter.getReadyState()){
			ret++;
		}
		return ret;
	}

}
